/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _GTOF_CORE_H_
#define _GTOF_CORE_H_

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Includes Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <stdint.h>
#include "fsl_common.h"
#include "dtest_ctrl.h"
#include "gtof.h"
#include "gtof_debug.h"
#include "genfsk_interface.h"

#ifndef NOT_USED
#define NOT_USED(__x) (void)(__x)
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  State Machine Section
///////////////////////////////////////////////////////////////////////////////////////////////////

struct gtof_fsm_ctx_;
typedef void (* gtof_fsm_event_handler_t) (struct gtof_fsm_ctx_ *ctx, uint32_t event, void *eventCtx);

/*
 * A transition table will be defined that way:

const gtof_fsm_event_handler_t transTable[MAX_STATES][MAX_EVENTS] =
{
                  EVT1      EVT2     EVT3
    [STATE1] = { handler1, handler2, handler3},
    [STATE2] = { handler1, handler2, handler3},
};
*/

/* Context for FSM execution */
typedef struct gtof_fsm_ctx_
{
    uint32_t currentState;
    const gtof_fsm_event_handler_t *currentTransRow;
} gtof_fsm_ctx_t;

#define GTOF_FSM_CHANGE_STATE_TO(fsm, transTable, newState) \
    GTOF_DEBUG_VALUE2(GTOF_DEBUG_EVT_STATE_TRANSITION, (fsm)->currentState, (newState)); \
    (fsm)->currentState = (uint32_t)(newState);                    \
    (fsm)->currentTransRow = (transTable)[(uint32_t)(newState)]; /* Optimize next access */

#define GTOF_FSM_PROCESS_EVENT(fsm, event, eventCtx) \
    (fsm)->currentTransRow[(uint32_t)event](fsm, event, eventCtx);

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Defines & Macros Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef TOF_LOG_ENABLED
#define TOF_LOG_ENABLED                                 (0)
#endif

#ifndef TOF_LOG_SIZE
#define TOF_LOG_SIZE                                    (257u)
#endif

#ifndef TOF_OVERWRITE_AGC_STEP
#define TOF_OVERWRITE_AGC_STEP                          (0)
#endif

/* Processing time for computing distance on anchor side after DONE has been sent.
 * (value measured via analyser)
 */
#define TOF_ANCHOR_FINAL_PROC_TIME_US (80U)
    
/* Duration of anchor processing to add to GTOF_RETRY_TIMEOUT_US when 2 subsequent timeouts occur on RD 
 * (value measured via analyser)
 */
#define GTOF_RETRY_TIMEOUT_OFFSET_US (300U)
    
/* Time elapsed between 2 measures when doing calibration.
 * No need to make it too small, it's better to collect measures over a longer period of time.
 */
#define TOF_CALIBRATION_DELAY_MS (30U)

/* Retry timeout in us */
#define GTOF_RETRY_TIMEOUT_US (600U)

/* Interval between 2 START_REQ */
#define GTOF_START_INTERVAL_US (1000U)

/* Delay used for StartTx to leave some time on the peer to switch to Rx mode */
#define TOF_TX_DELAY_TIME_US (180U)

/* Value of reflector abort timer: started to cover packet loss during calibration / start sequence
 * Must be greater than TOF_CALIBRATION_DELAY_MS
 */
#define GTOF_ABORT_TIMER (TOF_CALIBRATION_DELAY_MS + TOF_ANCHOR_FINAL_PROC_TIME_US + 10U)

/* Maximum number of retries supported by GTOF - must fit a 4 bits */
#define GTOF_MAXIMUM_RETRIES (0xFu)

/*
 * TOF packet Header is made of:
 *  0 1 2 3 4 5 6 7 8 9 A B C D E F
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |  RC   |  SN   |U|U|U|A| OpCode|
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * LL header:
 *   RC: retry count
 *   SN: Sequence Number
 * Tof header:
 *   A: Ack bit (aka Response bit)
 *   OpCode: identifier of Tof message
 * U:unused/reserved
 */
#define GTOF_RESP_BIT  (0x10u)
#define GTOF_RETRYCNT_MASK (0xF0u)
#define GTOF_RETRYCNT_SHIFT (4u)
#define GTOF_HDR_SEQNUM_MASK (0x0Fu)
#define GTOF_HDR_OPCODE_MASK (0x0Fu)
#define GTOF_RESP_OPCODE(opcode) ((opcode) | GTOF_RESP_BIT)
#define GTOF_HDR_GET_SEQNUM(pkt) ((pkt)->tofHdr.seqNum & GTOF_HDR_SEQNUM_MASK)
#define GTOF_HDR_SET_SEQNUM(pkt, SEQNUM) ((pkt)->tofHdr.seqNum = ((SEQNUM) & (uint8_t)GTOF_HDR_SEQNUM_MASK))
#define GTOF_HDR_GET_OPCODE(pkt) ((pkt)->tofHdr.opCode & GTOF_HDR_OPCODE_MASK)
#define GTOF_HDR_GET_RETRYCNT(pkt) (((pkt)->tofHdr.seqNum & (uint8_t)GTOF_RETRYCNT_MASK) >> GTOF_RETRYCNT_SHIFT)
#define GTOF_HDR_SET_RETRYCNT(pkt, retries) ((pkt)->tofHdr.seqNum |= (((retries)<<GTOF_RETRYCNT_SHIFT) & (uint8_t)GTOF_RETRYCNT_MASK))

/* WARNING: these SET/GET_LEN accelerators assume that TOF H0 len is 1 byte and H1 is set to zero) */
#if (GFSKTOF_H0_BIT_SIZE != 8u)
#error "GFSKTOF_H0_BIT_SIZE must be multiple of 8"
#endif
#define GTOF_HDR_SET_LEN(pkt, lenght) ((pkt)->gfskHdr[GFSKTOF_SYNC_ADDRESS_SIZE+1U+(GFSKTOF_H0_BIT_SIZE/8U)] = (uint8_t)((lenght) | (GFSKTOF_H1_VALUE << GFSKTOF_LENGHT_FIELD_SIZE)))
#define GTOF_HDR_GET_LEN(pkt) (((gtof_packet_t *)(void *)(pkt))->gfskHdr[GFSKTOF_SYNC_ADDRESS_SIZE+1U+(GFSKTOF_H0_BIT_SIZE/8U)])

/* OP codes used by control protocol on air interface */
#define TOF_START_MEASUREMENT_OPCODE                    (0x0Au)
#define TOF_DONE_OPCODE									(0x0Bu)
#define TOF_GET_TIMESTAMP_OPCODE						(0x0Cu)
#define TOF_START_CALIBRATION_OPCODE    				(0x0Du)
#define TOF_TIMESTAMP_RESPONSE_OPCODE                   (GTOF_RESP_OPCODE(TOF_GET_TIMESTAMP_OPCODE))

#define TOF_HOP_SEED_DEFAULT                            (0x7Fu)

/* Values for current Radio mode */
#define TOF_TX_MODE										(0u)
#define TOF_RX_MODE										(1u)

/* As COUNT_TO_NSEC has a mul and div, the fixed point remains on the same place */
#define COUNT_TO_NSEC(Count)                            (uint64_t)(((uint64_t)(Count)*1000000000u)/TofXcvrData.timerReferenceClockHZ)
#define MSEC_TO_NSEC(Msec)                              (uint64_t)(Msec*1000u)

#define GTOF_GFSK_PACK_HDR_SZ (GFSKTOF_SYNC_ADDRESS_SIZE + 1U + GFSKTOF_HEADER_SIZE)

/* Maximum payload size based on number of bits for lenght field */
#define GTOF_PAYLOAD_MAX_SZ ((1U<<GFSKTOF_LENGHT_FIELD_SIZE)-1U)
#define GTOF_PACKET_BUFFER_SZ (GTOF_GFSK_PACK_HDR_SZ + GTOF_PAYLOAD_MAX_SZ + GFSKTOF_CRCSIZE)

/* Flags used to control SendPacket API */
#define GTOF_PACK_CTRL_ACK   (1U<<0)
#define GTOF_PACK_CTRL_RETRY (1U<<1)
            
///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Typedef Section
///////////////////////////////////////////////////////////////////////////////////////////////////
        
/* Prototype for timestamp handling callbacks */
typedef void (*toftimestampCb_t)(uint32_t mode, uint16_t timestamp, uint8_t agcIndex);

/* Timestamp / XCVR control */
typedef struct tof_xcvr_data_
{
    uint32_t rxWarmupCount;
    uint32_t timerReferenceClockHZ;
    uint32_t txRxMode;
    uint32_t fracBitShift;                 /*!< number of bits that must be shifted to convert FRAC register */
    toftimestampCb_t timestampHandlingCb;  /*!< Rx/Tx timestamp handling callback*/
} tof_xcvr_data_t;

typedef struct {
    gtofMeasureConfig_t cfg;
    struct {
        uint8_t *pDeviceChannelMap;
        uint8_t deviceChannelsInMap;
        gtofAgcBehavior_t agcBehavior;
        uint8_t agcIndex;
        uint8_t maxAgcIndex;               /*!< Maximum AGC index that will be used by ToF */
    } deviceCfg;
    uint64_t syncTimestamp;
    uint16_t sampleId;
    uint8_t nbRetry;
    uint8_t seqNum;
    uint8_t deviceChannelMapIndex;
    uint8_t startConfirmed;
    uint8_t samplesPerFreqCnt;
} gtof_session_t;

/*
 * Warning: for all packet structs below, be careful to only use 1 byte lenght fields
 * not to screw up various sizeofs and/or alignements.
 */
typedef struct {
    uint8_t seqNum;
    uint8_t opCode;
} tofHdr_t;

#define GTOF_PKT_HDR_STRUCT \
    uint8_t gfskHdr[GTOF_GFSK_PACK_HDR_SZ]; \
    tofHdr_t tofHdr

typedef struct {
    GTOF_PKT_HDR_STRUCT;
    //uint8_t data[0];
} gtof_packet_t;

typedef struct {
    GTOF_PKT_HDR_STRUCT;
} gtof_packet_get_ts_t;

typedef struct {
    GTOF_PKT_HDR_STRUCT;
    uint8_t iterations;
} gtof_packet_start_calibration_t;

typedef struct {
    GTOF_PKT_HDR_STRUCT;
    uint8_t startSeqCounter;
    uint8_t maxDurationMs_msb;
    uint8_t maxDurationMs_lsb;
    uint8_t maxRetriesPerFreq;
    uint8_t maxSamplesPerFreq;
    uint8_t freqSeed;
    uint16_t nbSamples;
} gtof_packet_start_measurement_t;

typedef struct {
    GTOF_PKT_HDR_STRUCT;
    uint8_t t2_msb;
    uint8_t t2_lsb;
    uint8_t t3_msb;
    uint8_t t3_lsb;
    uint8_t agcIndex;
    int8_t rssi;
    int8_t cfoEstimate;
} gtof_packet_ts_response_t;

/* Descriptor of TOF_EVT_RX */
typedef struct gtof_evt_rx_descr_
{
    uint8_t *rxDataBuffer;
    int8_t rssiValue;
    int8_t cfoEstimate;
} gtof_evt_rx_descr_t;

/* Structure of Hop Log Record */
typedef struct {
    uint32_t currentHopIdx;
    uint32_t channelIdxAccu;
#if TOF_LOG_ENABLED == 1
    ToFChannelLogEntry_t entries[TOF_LOG_SIZE];
#endif
} ToFChannelLog_t;

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                Function-like Macros Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Extern Constants Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Extern Variables Section
///////////////////////////////////////////////////////////////////////////////////////////////////

/* ToF Configuration */
extern gtofAppConfig_t TofAppConfig;
extern gtofPeerConfig_t TofPeerConfig;

extern ToFChannelLog_t ChannelHopLog;
extern tof_xcvr_data_t TofXcvrData;

/* ToF current session state */
extern gtof_session_t TofSession;

/* Store packet header details for GENFSK session */
extern GENFSK_packet_t TofTxPacket;

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                Function Prototypes Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#if defined(__cplusplus)
extern "C" {
#endif // __cplusplus

void Tof_TimestampTimerInit(const gtofAppConfig_t *appConfig, toftimestampCb_t callback);
void Tof_TimestampTimerStart(void);
void Tof_TimestampTimerStop(void);
void Tof_XcvrCustomConfiguration(bool isEnabled);

uint8_t Tof_GetAgcIndex(void);
void Tof_LockAgc(uint8_t Index);
void Tof_UnlockAgc(void);

void Tof_FrequencyHopReset(void);
void Tof_FrequencyHopChangeChannel(void);

void Tof_SessionReset(gtof_session_t *session);
void Tof_StopRx(void);
genfskStatus_t Tof_SendPacket(uint8_t *pBuffer, uint16_t bufLengthBytes, uint32_t txDelay, uint32_t flags);

void Tof_TriggerEnableTx(void);
void Tof_TriggerEnableRx(void);
void Tof_TriggerDisable(void);

#if defined(__cplusplus)
}
#endif // __cplusplus


#endif /* _GTOF_CORE_H_ */
///////////////////////////////////////////////////////////////////////////////////////////////////
// EOF
///////////////////////////////////////////////////////////////////////////////////////////////////
